/*
    portscan match for iptables

    Copyright ? Jan Engelhardt <jengelh [at] gmx de>, 2006 - 2007
    released under the terms of the GNU General Public
    License version 2.x and only versions 2.x.
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <iptables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/netfilter/xt_portscan.h>

static void libipt_portscan_help(void)
{
	printf(
		"portscan match v%s options:\n"
		"(Combining them will make them match by OR-logic)\n"
		"  --stealth    Match TCP Stealth packets\n"
		"  --synscan    Match TCP SYN scans\n"
		"  --cnscan     Match TCP Connect scans\n"
		"  --grscan     Match Banner Grabbing scans\n",
		IPTABLES_VERSION);
	return;
}

static void libipt_portscan_mtinit(struct ipt_entry_match *match,
    unsigned int *nfcache)
{
	/* Cannot cache this */
	*nfcache |= NFC_UNKNOWN;
	return;
}


static int libipt_portscan_parse(int c, char **argv, int invert,
    unsigned int *flags, const struct ipt_entry *entry, unsigned int *nfc,
    struct ipt_entry_match **match)
{
	struct xt_portscan_info *info = (void *)((*match)->data);

	switch(c) {
		case 'c':
			info->match_cn = 1;
			*flags = 1;
			return 1;
		case 'g':
			info->match_gr = 1;
			*flags = 1;
			return 1;
		case 's':
			info->match_syn = 1;
			*flags = 1;
			return 1;
		case 'x':
			info->match_stealth = 1;
			*flags = 1;
			return 1;
		default:
			return 0;
	}
}

static void libipt_portscan_check(unsigned int flags)
{
	if (!flags)
		xtables_error(PARAMETER_PROBLEM, "You must specify `--stealth' or '--synscan' or '--cnscan' or '--grscan'");
}

static void libipt_portscan_print(const struct ipt_ip *ip,
    const struct ipt_entry_match *match, int numeric)
{
	const struct xt_portscan_info *info = (const void *)(match->data);
	const char *s = "";

	printf("portscan ");
	if(info->match_stealth) {
		printf("STEALTH");
		s = ",";
	}
	if(info->match_syn) {
		printf("%sSYNSCAN", s);
		s = ",";
	}
	if(info->match_cn) {
		printf("%sCNSCAN", s);
		s = ",";
	}
	if(info->match_gr)
		printf("%sGRSCAN", s);
	printf(" ");
	return;
}

static void libipt_portscan_save(const struct ipt_ip *ip,
    const struct ipt_entry_match *match)
{
	const struct xt_portscan_info *info = (const void *)(match->data);
	if(info->match_stealth)	printf("--stealth ");
	if(info->match_syn)	printf("--synscan ");
	if(info->match_cn)	printf("--cnscan ");
	if(info->match_gr)	printf("--grscan ");
	return;
}

static struct option libipt_portscan_opts[] = {
	{"stealth", 0, NULL, 'x'},
	{"synscan", 0, NULL, 's'},
	{"cnscan",  0, NULL, 'c'},
	{"grscan",  0, NULL, 'g'},
	{NULL},
};

static struct xtables_match libipt_portscan_info = {
	.name          = "portscan",
	.version       = XTABLES_VERSION,
	.family        = NFPROTO_IPV4,
	.size          = XT_ALIGN(sizeof(struct xt_portscan_info)),
	.userspacesize = XT_ALIGN(sizeof(struct xt_portscan_info)),
	.help          = libipt_portscan_help,
	.init          = libipt_portscan_mtinit,
	.parse         = libipt_portscan_parse,
	.final_check   = libipt_portscan_check,
	.print         = libipt_portscan_print,
	.save          = libipt_portscan_save,
	.extra_opts    = libipt_portscan_opts,
};

void _init(void)
{
	xtables_register_match(&libipt_portscan_info);
	return;
}